<?php
/**
 * WireGuard Manager
 * مدیریت کانفیگ‌های وایرگارد
 */

class WireGuardManager
{
    private array $config;
    private array $peers = [];
    private string $peersFile;
    
    public function __construct(array $config)
    {
        $this->config = $config;
        $this->peersFile = $config['storage']['peers_file'];
        
        // ایجاد دایرکتوری‌های مورد نیاز
        $dirs = [
            dirname($this->peersFile),
            $config['storage']['configs_dir'],
            $config['storage']['qrcodes_dir'],
        ];
        
        foreach ($dirs as $dir) {
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
        }
        
        $this->loadPeers();
    }
    
    private function loadPeers(): void
    {
        if (file_exists($this->peersFile)) {
            $data = file_get_contents($this->peersFile);
            $this->peers = json_decode($data, true) ?: [];
        }
    }
    
    private function savePeers(): void
    {
        file_put_contents($this->peersFile, json_encode($this->peers, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    public function generateKeyPair(): array
    {
        // تولید کلید خصوصی
        $privateKey = sodium_crypto_box_keypair();
        $privateKeyBytes = sodium_crypto_box_secretkey($privateKey);
        $publicKeyBytes = sodium_crypto_box_publickey($privateKey);
        
        // برای WireGuard باید از curve25519 استفاده کنیم
        // استفاده از روش ساده‌تر با exec
        $privateKey = trim(shell_exec('wg genkey 2>/dev/null') ?: $this->generateWgKey());
        $publicKey = trim(shell_exec("echo '$privateKey' | wg pubkey 2>/dev/null") ?: $this->derivePublicKey($privateKey));
        
        return [
            'private_key' => $privateKey,
            'public_key' => $publicKey,
        ];
    }
    
    private function generateWgKey(): string
    {
        // Fallback: تولید کلید 32 بایتی رندوم و تبدیل به base64
        return base64_encode(random_bytes(32));
    }
    
    private function derivePublicKey(string $privateKey): string
    {
        // این یک fallback است - در صورت نبود wg command
        // باید از libsodium استفاده کنیم
        $privateBytes = base64_decode($privateKey);
        if (strlen($privateBytes) !== 32) {
            throw new Exception("Invalid private key length");
        }
        
        // Curve25519 scalar multiplication
        $publicBytes = sodium_crypto_scalarmult_base($privateBytes);
        return base64_encode($publicBytes);
    }
    
    public function getNextAvailableIP(): ?string
    {
        $pool = $this->config['ip_pool'];
        $usedIPs = array_column($this->peers, 'ip');
        
        for ($i = $pool['start']; $i <= $pool['end']; $i++) {
            $ip = $pool['network'] . $i;
            if (!in_array($ip, $usedIPs)) {
                return $ip;
            }
        }
        
        return null;
    }
    
    public function createPeer(string $name, int $telegramUserId): array
    {
        // بررسی تکراری نبودن نام
        foreach ($this->peers as $peer) {
            if ($peer['name'] === $name) {
                throw new Exception("یک peer با این نام قبلاً وجود دارد: $name");
            }
        }
        
        // گرفتن IP جدید
        $ip = $this->getNextAvailableIP();
        if (!$ip) {
            throw new Exception("IP خالی در pool وجود ندارد");
        }
        
        // تولید کلیدها
        $keys = $this->generateKeyPair();
        
        $peer = [
            'id' => uniqid('peer_'),
            'name' => $name,
            'ip' => $ip,
            'private_key' => $keys['private_key'],
            'public_key' => $keys['public_key'],
            'telegram_user_id' => $telegramUserId,
            'created_at' => date('Y-m-d H:i:s'),
            'mikrotik_added' => false,
        ];
        
        $this->peers[$peer['id']] = $peer;
        $this->savePeers();
        
        return $peer;
    }
    
    public function generateConfig(array $peer): string
    {
        $wg = $this->config['wireguard'];
        
        $config = "[Interface]\n";
        $config .= "PrivateKey = {$peer['private_key']}\n";
        $config .= "Address = {$peer['ip']}/32\n";
        $config .= "DNS = {$wg['dns']}\n";
        $config .= "\n";
        $config .= "[Peer]\n";
        $config .= "PublicKey = {$wg['server_public_key']}\n";
        $config .= "AllowedIPs = {$wg['allowed_ips']}\n";
        $config .= "Endpoint = {$wg['server_endpoint']}:{$wg['server_port']}\n";
        $config .= "PersistentKeepalive = {$wg['persistent_keepalive']}\n";
        
        return $config;
    }
    
    public function saveConfigFile(array $peer): string
    {
        $config = $this->generateConfig($peer);
        $filename = $this->config['storage']['configs_dir'] . $peer['name'] . '.conf';
        file_put_contents($filename, $config);
        return $filename;
    }
    
    public function generateQRCode(array $peer): string
    {
        $config = $this->generateConfig($peer);
        $filename = $this->config['storage']['qrcodes_dir'] . $peer['name'] . '.png';
        
        // استفاده از کتابخانه phpqrcode یا endroid/qr-code
        // در اینجا از یک روش ساده با Google Charts API استفاده می‌کنیم
        // برای production بهتره از کتابخانه لوکال استفاده کنید
        
        $qrContent = urlencode($config);
        $qrUrl = "https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=" . $qrContent;
        
        $qrImage = file_get_contents($qrUrl);
        if ($qrImage) {
            file_put_contents($filename, $qrImage);
            return $filename;
        }
        
        throw new Exception("خطا در تولید QR Code");
    }
    
    public function getPeer(string $id): ?array
    {
        return $this->peers[$id] ?? null;
    }
    
    public function getPeerByName(string $name): ?array
    {
        foreach ($this->peers as $peer) {
            if ($peer['name'] === $name) {
                return $peer;
            }
        }
        return null;
    }
    
    public function getAllPeers(): array
    {
        return $this->peers;
    }
    
    public function getUserPeers(int $telegramUserId): array
    {
        return array_filter($this->peers, fn($peer) => $peer['telegram_user_id'] === $telegramUserId);
    }
    
    public function deletePeer(string $id): bool
    {
        if (isset($this->peers[$id])) {
            // حذف فایل‌ها
            $peer = $this->peers[$id];
            $configFile = $this->config['storage']['configs_dir'] . $peer['name'] . '.conf';
            $qrFile = $this->config['storage']['qrcodes_dir'] . $peer['name'] . '.png';
            
            if (file_exists($configFile)) unlink($configFile);
            if (file_exists($qrFile)) unlink($qrFile);
            
            unset($this->peers[$id]);
            $this->savePeers();
            return true;
        }
        return false;
    }
    
    public function markMikroTikAdded(string $id): void
    {
        if (isset($this->peers[$id])) {
            $this->peers[$id]['mikrotik_added'] = true;
            $this->savePeers();
        }
    }
    
    public function getStats(): array
    {
        $pool = $this->config['ip_pool'];
        $totalIPs = $pool['end'] - $pool['start'] + 1;
        $usedIPs = count($this->peers);
        
        return [
            'total_ips' => $totalIPs,
            'used_ips' => $usedIPs,
            'available_ips' => $totalIPs - $usedIPs,
            'total_peers' => count($this->peers),
        ];
    }
}
